unit lvm2;

interface

uses windows, classes, sysutils, dialogs, ex2;

type lvm2_pv_head1 = record
	s_name                : array [0..7] of CHAR; // some kind of name
   s_unknown1            : U64;
   s_unknown2            : U64;
	s_magic               : array [0..7] of CHAR;  // LVM2 001
	s_uuid                : array [0..31] of CHAR;  // UUID
   s_unknown3            : U64; // size of device ?
   s_unknown4            : U64;
   s_unknown5            : U64;
   s_unknown6            : U64;
   s_unknown7            : U64;
   s_unknown8            : U64;
   s_unknown9            : U64;
   s_unknown10           : U64;
   padding1              : array [0..127] of CHAR; // pad up to 512 bytes
   padding2              : array [0..255] of CHAR; // pad up to 512 bytes
end;
type Plvm2_pv_head1 = ^lvm2_pv_head1;

type lvm2_pv_head2 = record
   s_unknown1            : ULONG;
	s_name                : array [0..3] of CHAR;
   s_unknown2            : U64;
   s_unknown4            : U64;
   s_offset1             : U64;
   s_unknown7            : U64;
   s_offset2             : U64;
   s_len1                : U64;
   s_unknown8            : U64;
   padding1              : array [0..63] of CHAR; // pad up to 512 bytes
   padding2              : array [0..127] of CHAR; // pad up to 512 bytes
   padding3              : array [0..255] of CHAR; // pad up to 512 bytes
end;
type Plvm2_pv_head2 = ^lvm2_pv_head2;



type TMetadataNode = class(TObject)
  public

   Parent : TMetadataNode;
   Name : String;
   Keys : TStringList;
   Children : TList;

   constructor Create(Parent : TMetadataNode);

   function GetPath : String;
   procedure AddKey(Key : String; Value : String);
   function FindChild(Name : String) : TMetadataNode;
   function GetValue(Key : String) : String;
   function GetValueInt(Key : String) : Integer;

end;

type TLVMPV = class(TObject)
  public
   constructor Create(Device : String);
   function DevRead(Offset : Int64; Length : Integer; Buffer : Pointer) : Boolean;

   procedure ParseMetadata;
   procedure FindLV(PartitionList : TList);

  private
   Handle   : THANDLE;
   Device   : String;
   Metadata : String;
   MetadataTree : TMetadataNode;

   lvm_head1 : lvm2_pv_head1;
   lvm_head2 : lvm2_pv_head2;
   
end;

procedure LVM2AddPV(DevName : String);
procedure LVMBuildLV(PartitionList : TList);

implementation

uses ex2explore, native, winioctl, partition;

var
   PVDevices : TStringList;

procedure LVM2AddPV(DevName : String);
begin
   if PVDevices = nil then
   begin
      PVDevices := TStringList.Create;
   end;

   PVDevices.Add(DevName);
end;

procedure LVMBuildLV(PartitionList : TList);
var
   i : Integer;
   PV : TLVMPV;
begin
   if Assigned(PVDevices) then
   begin
      Debug('Scanning LVM2', DebugLow);

      for i := 0 to PVDevices.Count - 1 do
      begin
         PV := TLVMPV.Create(PVDevices[i]);
         PV.FindLV(PartitionList);
      end;
   end
   else
   begin
      Debug('No LVM2 detected', DebugLow);
   end;

end;

constructor TLVMPV.Create(Device : String);
var
   Offset   : Integer;
   Len      : Integer;
begin

   self.Device := Device;
   Handle := NTCreateFile(PChar(Device), GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING, FILE_FLAG_RANDOM_ACCESS, 0);


   if Handle <> INVALID_HANDLE_VALUE then
   begin
      // there is metadata at $200 -> $27F
      DevRead(512, sizeof(lvm_head1), @lvm_head1);

      // and more at $800 -> $83F
      DevRead(2048, sizeof(lvm_head2), @lvm_head2);

      Debug('metadata from ' + IntToStr(lvm_head2.s_offset1 + lvm_head2.s_offset2) + ' len ' + IntToStr(lvm_head2.s_len1), DebugHigh);

      Offset := lvm_head2.s_offset1 + lvm_head2.s_offset2;
      Len := lvm_head2.s_len1;

      if Offset mod 512 <> 0 then
      begin
        // and increase the len..
        Len := Len + (Offset mod 512);

        // move the offset backwards..
        Offset := Offset - (Offset mod 512);
      end;

      if Len mod 512 <> 0 then
      begin
        Len := Len + (512 - (Len mod 512));
      end;

      SetLength(Metadata, Len);

      DevRead(Offset, Len, PChar(Metadata));
      Metadata := AdjustLineBreaks(Metadata);
      Debug(Metadata, DebugHigh);

      ParseMetadata;
   end
   else
   begin
      Debug('Got invalid handle (error)', DebugOff);
   end;

end;

function TLVMPV.DevRead(Offset : Int64; Length : Integer; Buffer : Pointer) : Boolean;
var
   ReadStart  : _Large_Integer;
   Seek       : DWORD;
   Actual     : DWORD;
begin
   Result := False;
   ReadStart.QuadPart := Offset;
   Seek := SetFilePointer(Handle, Offset, nil, FILE_BEGIN);
   if Seek <> $FFFFFFFF then
   begin
      if ReadFile2(Handle, Buffer, Length, Actual, nil) then
      begin
         Result := True;
      end
      else
      begin
         Debug('Error reading device ' + IntToStr(GetLastError) + ' ' + SysErrorMessage(GetLastError), DebugHigh);
      end
   end
   else
   begin
      Debug('Seek failed error=' + IntToStr(GetLastError), DebugHigh);
   end;
end;

procedure TLVMPV.FindLV(PartitionList : TList);
var
   i : Integer;
   j : Integer;
   N : TMetadataNode;
   PV : TMetadataNode;
   LV : TMetadataNode;
   LVN : TMetadataNode;
   SegmentCount : Integer;
   Segment : TMetadataNode;
   Start : Integer;
   Len : Integer;
   ExtentSize : Int64;
   NewPart : TPartition;
begin
   if MetadataTree = nil then
   begin
      // can't do much with that...
      exit;
   end;

   // check the number of PV
   if MetadataTree.Children.Count = 1 then
   begin
      N := MetadataTree.Children[0];
      Debug('Extent size is ' + IntToStr(N.GetValueInt('extent_size') div 2) + 'k', DebugHigh);
      ExtentSize := N.GetValueInt('extent_size') * 512;

      PV := N.FindChild('physical_volumes');
      if Assigned(PV) and (PV.Children.Count = 1) then
      begin
         PV := PV.Children[0];
         // we could chek the ID
         // do we need pe_start & pe_count
         Debug('There are ' + IntToStr(PV.GetValueInt('pe_count')) + ' extents', DebugHigh);

         LV := N.FindChild('logical_volumes');
         if Assigned(LV) then
         begin
            Debug('There are ' + IntToStr(LV.Children.Count) + ' logical volumes', DebugHigh);

            for i := 0 to LV.Children.Count - 1 do
            begin
               LVN := LV.Children[i];
               Debug('LV id = ' + LVN.GetValue('id'), DebugHigh);

               SegmentCount := LVN.GetValueInt('segment_count');
               if SegmentCount = 1 then
               begin
                  for j := 1 to SegmentCount do // there is only 1
                  begin
                     Segment := LVN.FindChild('segment' + IntToStr(j));
                     if (Segment.GetValue('type') = 'striped') and
                        (Segment.GetValueInt('stripe_count') = 1) then
                     begin
                        // almost there...
                        // start = start_extent + stripe[2]
                        // len   = extent_count
                        Start := Segment.GetValueInt('start_extent') + Segment.GetValueInt('stripes[2]');
                        Len := Segment.GetValueInt('extent_count');
                        Debug('Start = ' + IntToStr(Start), DebugHigh);
                        Debug('Len   = ' + IntToStr(Len), DebugHigh);

                        NewPart := TPartition.Create;
                        NewPart.DriveNo                  := i;
                        NewPart.FileName                 := Device;
                        NewPart.DeviceName               := N.Name + '/' + LVN.Name;
                        NewPart.BytesPerSector           := 512;
                        NewPart.MediaType                := Media_Type_FixedMedia;
                        NewPart.PartitionType            := 0;//PARTITION_LINUX; // This will indicate that the fs has not yet been checked
                        NewPart.RecognizedPartition      := True;
                        NewPart.RewritePartition         := False;
                        NewPart.StartingOffset.QuadPart  := Start * ExtentSize + $30000;
                        NewPart.PartitionLength.QuadPart := Len * ExtentSize;
                        NewPart.HiddenSectors            := 0;
                        if Assigned(PartitionList) then
                        begin
                           PartitionList.Add(NewPart);
                        end;

                     end
                     else
                     begin
                        Debug('Wrong number of stripes (only 1 stripe supported)', DebugLow);
                     end;
                  end;
               end
               else
               begin
                  Debug('Wrong number of segments (only 1 supported)', DebugLow);
               end;
            end;
         end
         else
         begin
            Debug('Can''t find logical_volumes', DebugLow);
         end;
      end
      else
      begin
         Debug('Wrong number of physical volumes (only 1 supported)', DebugLow);
      end;
   end
   else
   begin
      Debug('Wrong number of volume groups', DebugLow);
   end;

   CloseHandle(Handle);
end;

procedure TLVMPV.ParseMetadata;
var
   L : String;
   N : TMetadataNode;
   TokenState : Integer;
   Token : String;
   TokenType : Integer;
   Tokens : TStringList;
   i : Integer;
   ParseState : Integer;
   Name : String;
   SetCount : Integer;

   procedure AcceptToken;
   begin
      if (Length(Token) > 0) or (TokenType = 1) then
      begin
         Tokens.Add(Token);
         Debug('T ' + Token, DebugHigh);
         Token := '';
         TokenType := 0; // 0 = unknown, 1 = string constant
      end;
   end;
begin
   // this is a heirachical key = value type config file

   // constraints...
   // must only have one physical_volumes (this object)
   // we can have many logical_volumes but they must only have one segment
   //  and that segment must be a linear stripe
   //  do some maths and work out the location inside this device and then we can
   //  create some sudo partitions.
   // once all that is done, rescan for ext2/3
   // simple eh?


   // parser...
   // we should write a real parser but for now I'll cheat and just try and
   // get it going
   Tokens := TStringList.Create;
   L := Metadata;
   TokenState := 0;
   for i := 1 to Length(L) do
   begin
      if TokenState = 0 then
      begin
         if L[i] = '"' then
         begin
            TokenState := 1; // in quote
         end
         else if L[i] = '#' then
         begin
            TokenState := 2; // in comment
         end
         else if L[i] in [ '[', ']', ',' ] then
         begin
            // set tokens
            AcceptToken;
            Token := L[i];
            AcceptToken;
         end
         else if L[i] in [' ', #10, #13] then
         begin
            AcceptToken;
         end
         else
         begin
            Token := Token + L[i];
         end;
      end
      else if TokenState = 1 then
      begin
         TokenType := 1; // string constant
         if L[i] = '"' then
         begin
            TokenState := 0; // back to normal
         end
         else
         begin
            Token := Token + L[i];
         end;
      end
      else if TokenState = 2 then
      begin
         if L[i] in [#10, #13] then
         begin
            TokenState := 0; // back to normal
         end;
      end;
   end;


   N := TMetadataNode.Create(nil);
   N.Name := '';

   // 0 = nil
   // 1 = name
   // 2 = {
   // 3 = =

   SetCount := 0;
   ParseState := 0;
   for i := 0 to Tokens.Count - 1 do
   begin
      Token := Tokens[i];
      if ParseState = 0 then
      begin
         if Token = '}' then
         begin
            // end of node
            if Assigned(N) then
            begin
               N := N.Parent;
            end;
            ParseState := 0;
         end
         else
         begin
            Name := Token;
            ParseState := 1;
         end;
      end
      else if ParseState = 1 then
      begin
         // expect { or =
         if Token = '{' then
         begin
            // start a new node
            N := TMetadataNode.Create(N);
            N.Name := Name;
            ParseState := 0;
         end
         else if Token = '=' then
         begin
            // read the value
            ParseState := 3;
         end
         else
         begin
            Debug('Parse Error. { or = expected but ' + Token + ' found', DebugLow);
         end;
      end
      else if ParseState = 3 then
      begin
         if Token = '[' then
         begin
            SetCount := 1;
            ParseState := 4;
         end
         else
         begin
            //Debug(N.GetPath + Name + ' = ' + Token, DebugOff);
            N.AddKey(Name, Token);
            ParseState := 0;
         end;
      end
      else if ParseState = 4 then
      begin
         // this is a set....
         //Debug(N.GetPath + Name + '[' + IntToStr(SetCount) + '] = ' + Token, DebugOff);
         N.AddKey(Name + '[' + IntToStr(SetCount) + ']', Token);
         ParseState := 5;
      end
      else if ParseState = 5 then
      begin
         if Token = ',' then
         begin
            SetCount := SetCount + 1;
            ParseState := 4;
         end
         else if Token = ']' then
         begin
            ParseState := 0;
         end
         else
         begin
            Debug('Parse Error. , or ] expected but ' + Token + ' found', DebugLow);
         end;
      end;
   end;

   if N.Parent = nil then
   begin
      MetadataTree := N;
   end
   else
   begin
      Debug('Incomplete metadata', DebugLow);
   end;

end;

constructor TMetadataNode.Create(Parent : TMetadataNode);
begin
   self.Parent := Parent;
   Children := TList.Create;
   if Assigned(Parent) then
   begin
      Parent.Children.Add(self);
   end;
   Keys := TStringList.Create;
end;

function TMetadataNode.GetPath : String;
begin
   Result := '';
   if Assigned(Parent) then
   begin
      Result := Parent.GetPath;
   end;
   Result := Result + Name + '\';
end;

procedure TMetadataNode.AddKey(Key : String; Value : String);
begin
   Keys.Add(Key + '=' + Value);
end;

function TMetadataNode.GetValue(Key : String) : String;
begin
   Result := Keys.Values[key];
end;

function TMetadataNode.GetValueInt(Key : String) : Integer;
begin
   try
      Result := StrToInt(Trim(Keys.Values[key]));
   except
      on E : Exception do
      begin
         ShowMessage(E.Message + #10 + 'While looking up ' + Name + '\' + Key);
         raise E;
      end;
   end;
end;

function TMetadataNode.FindChild(Name : String) : TMetadataNode;
var
   i : Integer;
begin
   Result := nil;
   for i := 0 to Children.Count - 1 do
   begin
      if TMetadataNode(Children[i]).Name = Name then
      begin
         Result := TMetadataNode(Children[i]);
         break;
      end
   end;
end;



end.
