unit RegUtil;
{
For example, let's say you want to save the last time your program was executed
              [using the Registry] and retrieve that value at the start of the program.

                   First, we have to get the current date and time as a string, so we can pass
                   it to the SaveStringToRegistry() function: 

              FormatDateTime(
                'mm/dd/yyyy hh:nn:ss', Now )



                   Now we have to come up with an unique Registry key (so that we can store
                   our values in the Registry without overwriting someone else's information).
                   Since we're storing our program specific information, we'll start with the
                   Software key and append the name of our company (called
                   MyCompanyName for example) and the name of our product (called
                   MyProgramName for example). SaveStringToRegistry() will automatically
                   append the key you pass to the HKEY_CURRENT_USER root key, which
                   will give you the following final key: 

                   HKEY_CURRENT_USER\Software\
                   MyCompanyName\MyProgramName

                   Finally, we're ready to make our call: 

              SaveStringToRegistry(
                //
                // our registry key
                //
                'Software'+
                '\MyCompanyName\MyProgramName', 

                //
                // parameter
                //
                'LastTime',

                //
                // value
                //
                FormatDateTime(
                  'mm/dd/yyyy hh:nn:ss', Now )
                );



                   To retrieve the information you've saved, simply call:

              sLastTime := 
                LoadStringFromRegistry(
                  'Software'+
                  '\MyCompanyName\MyProgramName', 

                  'LastTime',

                  //
                  // default value
                  //
                  'First Time'
                );



              If the parameter "LastTime" can not be found at the specified registry key,
              LoadStringFromRegistry() will return "First Time."

              Please note that if you have more than a few values to save and/or retrieve using
              the Registry, calling above functions over and over is not recommended. Rather,
              you should open the registry once, write/read all the information and then close
              the registry.
}

interface

uses Registry;

// Non OO functions
function LoadStringFromRegistry(sKey, sItem, sDefVal : string ) : string;
procedure SaveStringToRegistry(sKey, sItem, sVal : string );


// and a nice class to do everything at once
type
   TSoftwareRegistry = class(TObject)
   private
      Reg : TRegIniFile;
      Key : String;
   public
      constructor Create(Author : String; Product : String);
      destructor Destroy; override;

      function GetString(Item : String; Default : String) : String;
      procedure SetString(Item : String; Value : String);

      function GetBool(Item : String; Default : Boolean) : Boolean;
      procedure SetBool(Item : String; Value : Boolean);

      function GetInteger(Item : String; Default : LongInt) : Longint;
      procedure SetInteger(Item : String; Value : LongInt);

      procedure Notify;
   end;

implementation

function LoadStringFromRegistry(sKey, sItem, sDefVal : string ) : string;
var
   reg : TRegIniFile;
begin
   reg := TRegIniFile.Create( sKey );
   Result := reg.ReadString('', sItem, sDefVal );
   reg.Free;
end;

procedure SaveStringToRegistry(sKey, sItem, sVal : string );
var
   reg : TRegIniFile;
begin
   reg := TRegIniFile.Create( sKey );
   reg.WriteString('', sItem, sVal + #0 );
   reg.Free;
end;

constructor TSoftwareRegistry.Create(Author : String; Product : String);
begin
   Key := 'Software' + '\' + Author + '\' + Product;
   Reg := TRegIniFile.Create(Key);
end;

destructor TSoftwareRegistry.Destroy;
begin
   Reg.Free;
end;

function TSoftwareRegistry.GetString(Item : String; Default : String) : String;
begin
   GetString := Reg.ReadString('', Item, Default);
end;

procedure TSoftwareRegistry.SetString(Item : String; Value : String);
begin
   Reg.WriteString('', Item, Value + #0);  // What is the #0 ?
end;

function TSoftwareRegistry.GetBool(Item : String; Default : Boolean) : Boolean;
begin
   GetBool := Reg.ReadBool('', Item, Default);
end;

procedure TSoftwareRegistry.SetBool(Item : String; Value : Boolean);
begin
   Reg.WriteBool('', Item, Value);
end;

function TSoftwareRegistry.GetInteger(Item : String; Default : LongInt) : Longint;
begin
   GetInteger := Reg.ReadInteger('', Item, Default);
end;

procedure TSoftwareRegistry.SetInteger(Item : String; Value : LongInt);
begin
   Reg.WriteInteger('', Item, Value);
end;

procedure TSoftwareRegistry.Notify;
begin
(*   SendMessage( HWND_BROADCAST,
                WM_WININICHANGE,
                0,
                LongInt( PChar( Key ) )
              );*)
end;

end.
