unit DiskIO;

{$I ..\jgoops.i}

interface

uses QTThunkU, Windows;

type
  {$A-}
  TSectorInfo = record
    Drive      : BYTE;
    Cylinder   : WORD;
    Head       : BYTE;
    Sector     : BYTE;
    Count      : BYTE;
  end;

  TBlockInfo = record
    Drive   : BYTE;
    Block   : TLargeInteger;
    Count   : WORD;
  end;
  {$A+}

  PBlockInfo = ^TBlockInfo;

  T95Disk = class
  public
    constructor Create(EnableEI13 : Boolean);
    destructor Destroy; override;
    function SetDisk(Disk : Integer) : Boolean;
    procedure ExtractResourceToFile(Name : String; FileName : String);

    function ReadSector(SectorNo : LongInt; Buffer : Pointer; Count : LongInt) : Boolean;
    function WriteSector(SectorNo : LongInt; Buffer : Pointer; Count : LongInt) : Boolean;

    function EI13ReadSector(SectorNo : LongInt; Buffer : Pointer; Count : LongInt) : Boolean;
    function EI13WriteSector(SectorNo : LongInt; Buffer : Pointer; Count : LongInt) : Boolean;
    function EI13GetGeometry(Info : PBlockInfo) : Boolean;

    function SectorCount : LongInt;

    function DoRead(Sector : Integer; Head : Integer; Cylinder : Integer; Buffer : Pointer; Count : Integer) : Boolean;
    function DoWrite(Sector : Integer; Head : Integer; Cylinder : Integer; Buffer : Pointer; Count : Integer) : Boolean;

    function ResetDisk : Boolean;

    function GetGeometry : TSectorInfo;
  private
    UseEI13    : Boolean;
    Ready      : Boolean;
    DLLHandle  : THandle16;
    Geometry   : TSectorInfo;
    EI13Geometry : TBlockInfo;
  end;

implementation

uses ex2explore, SysUtils, classes, dialogs, forms, WinBinFile;

constructor T95Disk.Create(EnableEI13 : Boolean);
var
   Path : String;
{$ifdef JGO}
const
DLLNAME = 'Diskio2.dll';
//ha ha! I thought he used the wrong name! Ah child I am....
{$endif}
begin
   Path := ExtractFilePath(Application.ExeName) +
{$ifdef JGO}
   DLLNAME
{$else}
   'Diskio2.DLL'
{$endif}
   ;
   if not FileExists(Path) then
   begin
      // see if we have it as a resource
      ExtractResourceToFile('diskio', Path);
   end;

   try
      DLLHandle := LoadLib16(Path);
      Ready := False;
      UseEI13 := EnableEI13;
   except
      on E : EFOpenError do
      begin
         MessageDlg('Failed to load '+
{$ifdef JGO}
         DLLNAME
{$else}
         'diskio2.dll'
{$endif}
         +'.  Please make sure that this file is available (in the same directory as this application)', mtError, [mbCancel], 0);
         raise;
      end;
   end;
end;

destructor T95Disk.Destroy;
begin
   FreeLibrary16(DllHandle);
end;

function T95Disk.SetDisk(Disk : Integer) : Boolean;
begin
   if Disk < $80 then
   begin
      UseEI13 := False;
   end;
   if UseEI13 then
   begin
      EI13Geometry.Drive := Disk;
//      r := Call16BitRoutine('EI13GETDRIVEPARAMETERS', DllHandle, ccPascal,
//                            [@EI13Geometry], [sizeof(EI13Geometry)]);
      if EI13GetGeometry(@EI13Geometry) then
      begin
         Result := True;
         Ready := True;
      end
      else
      begin
         Result := False;
      end;
   end
   else
   begin
      Geometry.Drive := Disk;
      Geometry.Sector := 0;
      Call16BitRoutine('READDISKGEOMETRY', DllHandle, ccPascal,
                            [@Geometry], [sizeof(Geometry)]);
      if Geometry.Sector = 0 then
      begin
         Result := False;
      end
      else
      begin
         Result := True;
         Ready := True;
      end;
   end;
end;

procedure T95Disk.ExtractResourceToFile(Name : String; FileName : String);
var
   h : THandle;
   gh : THandle;
   data : PChar;
   BinFile : TBinaryFile;

begin
   h := FindResource(0, PChar(Name), 'DLL');
   if h > 0 then
   begin
      gh := LoadResource(0, h);
      if gh > 0 then
      begin
         Data := LockResource(gh);
         if Data <> nil then
         begin
            BinFile := TBinaryFile.Create;
            try
               BinFile.Assign(FileName);
               BinFile.Delete;
               BinFile.CreateNew;

               BinFile.BlockWrite2(Data, SizeofResource(0, h));
               BinFile.Close;
            finally
               BinFile.Free;
            end;
         end
         else
         begin
            // error
         end;
      end
      else
      begin
         // error
      end;
   end
   else
   begin
      // error
   end;
end;



function T95Disk.ResetDisk : Boolean;
var
   Pram  : TSectorInfo;
   R     : LongInt;
begin
   if Ready then
   begin
      if not UseEI13 then
      begin
         Pram.Drive  := Geometry.Drive;

         r := Call16BitRoutine('RESETDISK', DllHandle, ccPascal,
                               [@Pram], [sizeof(Pram)]);
         if r > 0 then
         begin
            Result := True;
         end
         else
         begin
            Result := False;
         end;
      end
      else
      begin
         Result := True; // just say true
      end;
   end
   else
   begin
      Result := False;
   end;
end;

function T95Disk.DoRead(Sector : Integer; Head : Integer; Cylinder : Integer; Buffer : Pointer; Count : Integer) : Boolean;
var
   Pram  : TSectorInfo;
   R     : LongInt;
begin
   if Ready then
   begin
      Pram.Drive  := Geometry.Drive;
        Pram.Sector := Sector;
        Pram.Head   := Head;
        Pram.Cylinder := Cylinder;
      Pram.Count  := Count;

      r := Call16BitRoutine('READPHYSICALSECTOR', DllHandle, ccPascal,
                            [@Pram, Buffer, Count * 512], [sizeof(Pram), Count * 512, 4]);
      if r > 0 then
      begin
         Result := True;
      end
      else
      begin
         Result := False;
      end;
   end
   else
   begin
      Result := False;
   end;
end;

function T95Disk.DoWrite(Sector : Integer; Head : Integer; Cylinder : Integer; Buffer : Pointer; Count : Integer) : Boolean;
var
   Pram  : TSectorInfo;
   R     : LongInt;
begin
   if Ready then
   begin
      Pram.Drive  := Geometry.Drive;
        Pram.Sector := Sector;
        Pram.Head   := Head;
        Pram.Cylinder := Cylinder;
      Pram.Count  := Count;

      r := Call16BitRoutine('WRITEPHYSICALSECTOR', DllHandle, ccPascal,
                            [@Pram, Buffer, 512 * Count], [sizeof(Pram), 512 * Count, 4]);
      if r > 0 then
      begin
         Result := True;
      end
      else
      begin
         Result := False;
      end;
   end
   else
   begin
      Result := False;
   end;
end;

function T95Disk.ReadSector(SectorNo : LongInt; Buffer : Pointer; Count : LongInt) : Boolean;
var
   Sector      : Integer;
   Head        : Integer;
   Cylinder    : Integer;
   Remainder   : Integer;

   BlockCount  : Integer;
begin
   if UseEI13 then
   begin
      Result := True;
      while Count > 0 do
      begin
         BlockCount := Count;
         if BlockCount > 127 then
         begin
            BlockCount := 127;
         end;

         Result := EI13ReadSector(SectorNo, Buffer, BlockCount);
         if Result = False then
         begin
            break;
         end;
         SectorNo := SectorNo + BlockCount;
         Count := Count - BlockCount;
         Buffer := @PChar(Buffer)[EI13Geometry.Count * BlockCount];
      end;
   end
   else
   begin
      Result := True;
      while Count > 0 do
      begin
         // xlate into chs
         Sector    := SectorNo mod Geometry.Sector;
         Remainder := SectorNo div Geometry.Sector;
        Head      := Remainder mod (Geometry.Head + 1);
        Cylinder  := Remainder div (Geometry.Head + 1);

         // see how many sectors ther are till the end of the track.....
         BlockCount := Geometry.Sector - Sector;
         if BlockCount > 255 then // max size...
         begin
            BLockCount := 255;
         end;

         if Count > BlockCount then
         begin
            if not DoRead(Sector + 1, Head, Cylinder, Buffer, BlockCount) then
            begin
               Result := False;
               break;
            end;
            Buffer := @PChar(Buffer)[512 * BlockCount]; // inc to next sector?
            SectorNo := SectorNo + BlockCount;
            Count := Count - BlockCount;
         end
         else
         begin
            if not DoRead(Sector + 1, Head, Cylinder, Buffer, Count) then
            begin
               Result := False;
               break;
            end;
            Count := 0;
         end;
      end;
   end;
end;

function T95Disk.WriteSector(SectorNo : LongInt; Buffer : Pointer; Count : LongInt) : Boolean;
var
   Sector      : Integer;
   Head        : Integer;
   Cylinder    : Integer;
   Remainder   : Integer;

   BlockCount  : Integer;
begin
   if UseEI13 then
   begin
      Result := True;
      while Count > 0 do
      begin
         BlockCount := Count;
         if BlockCount > 127 then
         begin
            BlockCount := 127;
         end;

         Result := EI13WriteSector(SectorNo, Buffer, BlockCount);
         if Result = False then
         begin
            break;
         end;
         SectorNo := SectorNo + BlockCount;
         Count := Count - BlockCount;
         Buffer := @PChar(Buffer)[EI13Geometry.Count * BlockCount];
      end;
   end
   else
   begin
      Result := True;
      while Count > 0 do
      begin
         // xlate into chs
         Sector    := SectorNo mod Geometry.Sector;
         Remainder := SectorNo div Geometry.Sector;
        Head      := Remainder mod (Geometry.Head + 1);
        Cylinder  := Remainder div (Geometry.Head + 1);

         // see how many sectors ther are till the end of the track.....
         BlockCount := Geometry.Sector - Sector;
         if BlockCount > 255 then // max size...
         begin
            BLockCount := 255;
         end;

         if Count > BlockCount then
         begin
            if not DoWrite(Sector + 1, Head, Cylinder, Buffer, BlockCount) then
            begin
               Result := False;
               break;
            end;
            Buffer := @PChar(Buffer)[512 * BlockCount]; // inc to next sector?
            SectorNo := SectorNo + BlockCount;
            Count := Count - BlockCount;
         end
         else
         begin
            if not DoWrite(Sector + 1, Head, Cylinder, Buffer, Count) then
            begin
               Result := False;
               break;
            end;
            Count := 0;
         end;
      end;
   end;
end;

function T95Disk.SectorCount : LongInt;
begin
   if Ready then
   begin
      Result := Geometry.Sector * (Geometry.Head + 1) * (Geometry.Cylinder + 1);
   end
   else
   begin
      Result := 0;
   end;
end;

function T95Disk.GetGeometry : TSectorInfo;
begin
   Result := Geometry;
end;

function T95Disk.EI13ReadSector(SectorNo : LongInt; Buffer : Pointer; Count : LongInt) : Boolean;
var
   Pram  : TBlockInfo;
   R     : LongInt;
begin
   Pram.Drive := EI13Geometry.Drive;
   Pram.Block := SectorNo;
   Pram.Count := Count;

   Debug('EI13ReadSector', DebugHigh);
   Debug('Drive ' + IntToStr(Pram.Drive), DebugHigh);
   Debug('SectorsLo = ' + IntToStr(Pram.Block), DebugHigh);
{$ifdef JGO}
   debug('Bytes = ' + IntToStr(Pram.Count * 512), DebugHigh);
{$else}
   Debug('Bytes per sector = ' + IntToStr(Pram.Count), DebugHigh);
{$endif}

   r := Call16BitRoutine('EI13READSECTOR', DllHandle, ccPascal,
                         [@Pram, Buffer, Count * 512], [sizeof(Pram), Count * 512, 4]);
   if r > 0 then
   begin
      Result := True;
   end
   else
   begin
      Result := False;
   end;
end;

function T95Disk.EI13WriteSector(SectorNo : LongInt; Buffer : Pointer; Count : LongInt) : Boolean;
var
   Pram  : TBlockInfo;
   R     : LongInt;
begin
   Pram.Drive := EI13Geometry.Drive;
   Pram.Block := SectorNo;
   Pram.Count := Count;

   Debug('EI13WriteSector', DebugHigh);
   Debug('Drive ' + IntToStr(Pram.Drive), DebugHigh);
   Debug('SectorsLo = ' + IntToStr(Pram.Block), DebugHigh);
   Debug('Bytes per sector = ' + IntToStr(Pram.Count), DebugHigh);


   r := Call16BitRoutine('EI13WRITESECTOR', DllHandle, ccPascal,
                         [@Pram, Buffer, Count * 512], [sizeof(Pram), Count * 512, 4]);
   if r > 0 then
   begin
      Result := True;
   end
   else
   begin
      Result := False;
   end;
end;

function T95Disk.EI13GetGeometry(Info : PBlockInfo) : Boolean;
var
   R     : LongInt;
begin
      Debug('EI13GetGeometry', debugHigh);

      r := Call16BitRoutine('EI13GETDRIVEPARAMETERS', DllHandle, ccPascal,
                            [Info], [sizeof(TBlockInfo)]);
      if r > 0 then
      begin
         Result := True;
         Debug('Drive ' + IntToStr(Info.Drive), debugHigh);
         Debug('SectorsLo = ' + IntToStr(Info.Block), debugHigh);
         Debug('Bytes per sector = ' + IntToStr(Info.Count), debugHigh);
      end
      else
      begin
         Result := False;
         Debug('Error', debugHigh);
      end;
end;


end.
